#!/usr/bin/env bash
# install_visualino_fedora.sh
# Instalador de Visualino 0.7.3 para Fedora desde el paquete tipo .deb desempaquetado
# Estructura esperada: directorio con subcarpeta ./usr/ (contiene binarios, íconos, .desktop, etc.)

set -euo pipefail

### ---- utilidades ----
msg() { printf "\n\033[1;36m[INFO]\033[0m %s\n" "$*"; }
ok()  { printf "\033[1;32m[OK]\033[0m %s\n" "$*\n"; }
err() { printf "\033[1;31m[ERROR]\033[0m %s\n" "$*\n" >&2; }
trap 'err "Fallo en la línea $LINENO. Revisa el mensaje anterior."' ERR

require_cmd() {
  command -v "$1" >/dev/null 2>&1 || { err "No se encontró el comando '$1'"; exit 127; }
}

### ---- comprobaciones previas ----
require_cmd dnf
require_cmd grep
require_cmd awk
require_cmd id

if [[ ! -d "usr" ]]; then
  err "No se encontró el directorio ./usr. Ejecuta este script dentro de la carpeta descomprimida de visualino_0.7.3_amd64."
  exit 1
fi

# Elevar a root si hace falta
if [[ $EUID -ne 0 ]]; then
  msg "Necesito privilegios de administrador. Reintentando con sudo..."
  exec sudo -E bash "$0" "$@"
fi

### ---- instalar dependencias ----
msg "Actualizando metadatos de DNF..."
dnf -y makecache

# Dependencias típicas de Visualino + Qt WebKit (donde está libQt5WebKitWidgets.so.5)
DEPS=(
  qt5-qtwebkit          # proporciona libQt5WebKitWidgets.so.5
  qt5-qtbase-gui
  qt5-qtsvg
  qt5-qtserialport
  desktop-file-utils
)

msg "Instalando dependencias requeridas: ${DEPS[*]}"
if ! dnf -y install "${DEPS[@]}"; then
  err "No se pudieron instalar todas las dependencias. En particular, si 'qt5-qtwebkit' no está disponible en tu versión de Fedora, es necesario habilitar un repositorio que lo provea o compilar QtWebKit manualmente."
  err "Opciones: buscar un repo compatible para qt5-qtwebkit o considerar un contenedor/AppImage. Continuaré con la instalación de archivos de Visualino."
fi

### ---- copiar archivos del paquete ----
msg "Copiando contenido de ./usr al sistema (/usr)..."
cp -rT usr /usr

# Reparar permisos básicos (por si vinieran incorrectos)
if [[ -f /usr/bin/visualino ]]; then
  chmod 0755 /usr/bin/visualino || true
fi

### ---- actualizar cachés de escritorio e íconos (si existen) ----
if command -v update-desktop-database >/dev/null 2>&1; then
  msg "Actualizando base de datos de entradas de escritorio..."
  update-desktop-database || true
fi

if [[ -d /usr/share/icons/hicolor ]] && command -v gtk-update-icon-cache >/dev/null 2>&1; then
  msg "Actualizando caché de íconos (hicolor)..."
  gtk-update-icon-cache -q /usr/share/icons/hicolor || true
fi

### ---- verificación de bibliotecas ----
if [[ -x /usr/bin/visualino ]]; then
  msg "Verificando dependencias dinámicas de /usr/bin/visualino..."
  if command -v ldd >/dev/null 2>&1; then
    # Mostrar solo faltantes, si los hay
    if ldd /usr/bin/visualino | grep -q "not found"; then
      ldd /usr/bin/visualino | grep "not found" || true
      err "Aún faltan bibliotecas. Revisa especialmente 'libQt5WebKitWidgets.so.5'."
      if rpm -q qt5-qtwebkit >/dev/null 2>&1; then
        msg "qt5-qtwebkit está instalado. Listando sus archivos para confirmar:"
        rpm -ql qt5-qtwebkit | grep -E 'libQt5WebKitWidgets\.so|libQt5WebKit\.so' || true
      else
        err "El paquete qt5-qtwebkit no está instalado. Intenta:  sudo dnf install qt5-qtwebkit"
      fi
      msg "Si la biblioteca existe pero en otra ruta no estándar, puedes exportar temporalmente LD_LIBRARY_PATH antes de ejecutar Visualino, p. ej.:
  export LD_LIBRARY_PATH=/usr/lib64:\$LD_LIBRARY_PATH
  visualino"
    else
      ok "Dependencias resueltas."
    fi
  fi
else
  err "No se encontró /usr/bin/visualino tras copiar archivos. Revisa la carpeta de origen ./usr/bin/."
  exit 1
fi

### ---- prueba de ejecución (opcional/no bloqueante) ----
msg "Probando ejecución no interactiva (esto puede mostrar logs de Qt y salir)..."
if /usr/bin/visualino >/dev/null 2>&1 & then
  sleep 1
  pkill -f "/usr/bin/visualino" || true
  ok "Visualino se lanzó correctamente (prueba rápida)."
else
  msg "Visualino no se inició en la prueba silenciosa. Intenta lanzarlo desde el menú o con:  /usr/bin/visualino"
fi

ok "Instalación finalizada."
echo
echo "Consejos:"
echo " - Puedes buscar Visualino en el menú de aplicaciones, o ejecutar: visualino"
echo " - Si sigues viendo 'libQt5WebKitWidgets.so.5: not found', asegúrate de que 'qt5-qtwebkit' esté instalado."
